/*
 * linux/drivers/serial/serial_s3c2440.c
 */

#include <linux/config.h>
#include <linux/module.h>
#include <linux/errno.h>
#include <linux/signal.h>
#include <linux/sched.h>
#include <linux/interrupt.h>
#include <linux/tty.h>
#include <linux/tty_flip.h>
#include <linux/serial.h>
#include <linux/major.h>
#include <linux/ptrace.h>
#include <linux/ioport.h>
#include <linux/mm.h>
#include <linux/slab.h>
#include <linux/init.h>
#include <linux/console.h>

#include <linux/serial.h>
#include <linux/serialP.h>
#include <linux/serial_reg.h>

#include <asm/io.h>
#include <asm/irq.h>
#include <asm/uaccess.h>
#include <asm/hardware.h>
#include <asm/serial.h>

#define	UTRSTAT_RX_RDY		1
#define	UTRSTAT_TX_RDY		(1<<1)

#define	UERSTAT_OVR_ERR		1
#define	UERSTAT_PAR_ERR		(1<<1)
#define	UERSTAT_FRM_ERR		(1<<2)

#define	UFSTAT_TX_FULL		(1<<14)
#define	UFSTAT_RX_CNT		0x1f

#define SERIAL_S3C2440_NAME	"ttyLU"
#define SERIAL_S3C2440_MAJOR	204
#define SERIAL_S3C2440_MINOR	0
#define WBUF_SIZE		1024

/* for DEBUG use */
static unsigned long UART_IntCount   = 0L;
static unsigned long UART_TxIntCount = 0L;
static unsigned long UART_RxIntCount = 0L;
static unsigned long UART_TxCount    = 0L;
static unsigned long UART_XmitCount  = 0L;

static int		use_count;
static unsigned short	flag;
static char		*putp;
static char		*getp;
static char		wbuf[WBUF_SIZE];

static struct tty_driver rs_driver;
static int		 rs_refcount;
static struct tty_struct *tty_table;
static struct termios	 *rs_termios;
static struct termios	 *rs_termios_locked;

static int rs_write_room(struct tty_struct *tty)
{
	if (putp >= getp)
		return (WBUF_SIZE - (int)putp + (int)getp);
	else
		return ((int)getp - (int)putp - 1);
}

static void rs_interrupt(int irq, void *dev_id, struct pt_regs *regs)
{
	unsigned long irqstat;
	int debug_flag = 0;

	UART_IntCount++;

	irqstat = IO_SUBSRCPND & ~IO_INTSUBMSK;
	if (irqstat & INT_RXD0) {
		UART_RxIntCount++;
		if (!flag) {
			IO_INTSUBMSK |= INT_RXD0;
		}
		else {
			while (IO_UTRSTAT0 & UTRSTAT_RX_RDY) {
				unsigned char c = IO_URXH0;
				tty_insert_flip_char(tty_table, c & 0xff, 0);
			}
			tty_flip_buffer_push(tty_table);
		}
		IO_SUBSRCPND = INT_RXD0;
	}
	if (irqstat & INT_TXD0) {
		UART_TxIntCount++;
		if (putp != getp) {
			IO_UTXH0 = *getp++;
			UART_XmitCount++;
			if (getp == wbuf + WBUF_SIZE)
				getp = wbuf;
			if (getp == putp)
				IO_INTSUBMSK |= INT_TXD0;
			if (flag)
				wake_up_interruptible(&tty_table->write_wait);
		}
		else {
			IO_INTSUBMSK |= INT_TXD0;
		}
		IO_SUBSRCPND = INT_TXD0;
	}
	IO_SRCPND = (1 << IRQ_UART0);
	IO_INTPND = (1 << IRQ_UART0);
}

static inline int rs_xmit(int c)
{
	if (putp + 1 == getp || ((putp + 1 == wbuf + WBUF_SIZE) && (getp == wbuf)))
		return 0;
	*putp = c;
	if (++putp >= wbuf + WBUF_SIZE)
		putp = wbuf;
	return 1;
}

static int rs_write(struct tty_struct *tty, int from_user, const u_char *buf, int count)
{
	int i;
	char c;

	if (from_user && verify_area(VERIFY_READ, buf, count))
		return -EINVAL;

	UART_TxCount += count;

	if (from_user) {
		for (i = 0; i < count; i++) {
			__get_user(c, buf + i);
			if (!rs_xmit(c))
				break;
		}
	}
	else {
		for (i = 0; i < count; i++) {
			c = buf[i];
			if (!rs_xmit(c))
				break;
		}
	}
	IO_INTSUBMSK &= ~INT_TXD0;
	return i;
}

static void rs_put_char(struct tty_struct *tty, u_char c)
{
	UART_TxCount++;
	rs_xmit(c);
	IO_INTSUBMSK &= ~INT_TXD0;
}

static int rs_chars_in_buffer(struct tty_struct *tty)
{
	return (WBUF_SIZE - rs_write_room(tty));
}

static void rs_flush_buffer(struct tty_struct *tty)
{
	IO_INTSUBMSK |= INT_TXD0;
	putp = wbuf;
	getp = wbuf;
}

static void rs_set_termios(struct tty_struct *tty, struct termios *old)
{
}

static void rs_stop(struct tty_struct *tty)
{
	IO_INTSUBMSK |= (INT_RXD0 | INT_TXD0);
}

static void rs_start(struct tty_struct *tty)
{
	IO_INTSUBMSK &= ~INT_RXD0;
}

static void rs_wait_until_sent(struct tty_struct *tty, int timeout)
{
	int orig_jiffies = jiffies;

	while (!(IO_UTRSTAT0 & UTRSTAT_TX_RDY)) {
		current->state = TASK_INTERRUPTIBLE;
		schedule_timeout(1);
		if (signal_pending(current))
			break;
		if (timeout && time_after(jiffies, orig_jiffies + timeout))
			break;
	}
	current->state = TASK_RUNNING;
}

static int rs_open(struct tty_struct *tty, struct file *filp)
{
	MOD_INC_USE_COUNT;
	tty->driver_data = NULL;
	flag++;
	IO_INTSUBMSK &= ~INT_RXD0;
	use_count++;
	return 0;
}

static void rs_close(struct tty_struct *tty, struct file *filp)
{
	if (!--use_count) {
		rs_wait_until_sent(tty, 0);
		IO_INTSUBMSK |= (INT_RXD0 | INT_TXD0);
		flag = 0;
	}
	MOD_DEC_USE_COUNT;
}

static int __init rs_init(void)
{
	memset(&rs_driver, 0, sizeof(rs_driver));
	rs_driver.magic			= TTY_DRIVER_MAGIC;
	rs_driver.driver_name		= "serial_s3c2440";
	rs_driver.name			= "ttyLU";
	rs_driver.name			= SERIAL_S3C2440_NAME;
	rs_driver.major			= SERIAL_S3C2440_MAJOR;
	rs_driver.minor_start		= SERIAL_S3C2440_MINOR;
	rs_driver.num			= 1;
	rs_driver.type			= TTY_DRIVER_TYPE_SERIAL;
	rs_driver.subtype		= SERIAL_TYPE_NORMAL;
	rs_driver.init_termios		= tty_std_termios;
	rs_driver.init_termios.c_cflag	= B115200 | CS8 | CREAD | HUPCL | CLOCAL;
	rs_driver.flags			= TTY_DRIVER_REAL_RAW;
	rs_driver.refcount		= &rs_refcount;
	rs_driver.table			= &tty_table;
	rs_driver.termios		= &rs_termios;
	rs_driver.termios_locked	= &rs_termios_locked;

	rs_driver.open			= rs_open;
	rs_driver.close			= rs_close;
	rs_driver.write			= rs_write;
	rs_driver.put_char		= rs_put_char;
	rs_driver.flush_chars		= NULL;
	rs_driver.write_room		= rs_write_room;
	rs_driver.chars_in_buffer	= rs_chars_in_buffer;
	rs_driver.flush_buffer		= rs_flush_buffer;
	rs_driver.ioctl			= NULL;
	rs_driver.throttle		= NULL;
	rs_driver.unthrottle		= NULL;
	rs_driver.send_xchar		= NULL;
	rs_driver.set_termios		= rs_set_termios;
	rs_driver.stop			= rs_stop;
	rs_driver.start			= rs_start;
	rs_driver.hangup		= NULL;
	rs_driver.break_ctl		= NULL;
	rs_driver.wait_until_sent	= rs_wait_until_sent;
	rs_driver.read_proc		= NULL;

	use_count = 0;
	flag = 0;
	putp = wbuf;
	getp = wbuf;
	if (request_irq(IRQ_UART0, rs_interrupt, SA_INTERRUPT, "serial", NULL))
		panic("Couldn't get S3C2440 serial irq\n");

	if (tty_register_driver(&rs_driver))
		panic("Couldn't register S3C2440 serial driver\n");

	return 0;
}

static void __exit rs_fini(void)
{
	unsigned long flags;
	int ret;

	save_flags(flags);
	cli();
	ret = tty_unregister_driver(&rs_driver);
	if (ret)
		printk(KERN_ERR "Unable to unregister S3C2440 serial driver (%d)\n", ret);
	free_irq(IRQ_UART0, NULL);
	restore_flags(flags);
}

module_init(rs_init);
module_exit(rs_fini);


#ifdef CONFIG_SERIAL_S3C2440_CONSOLE

static void rs_console_write(struct console *co, const char *s, u_int count)
{
	int i;
	unsigned long intmask;

	intmask = IO_INTSUBMSK & INT_TXD0;
	if (!intmask)
		IO_INTSUBMSK |= INT_TXD0;

	for (i = 0; i < count; i++) {
		while (!(IO_UTRSTAT0 & UTRSTAT_TX_RDY)) {}
		IO_UTXH0 = s[i];
		if (s[i] == '\n') {
			while (!(IO_UTRSTAT0 & UTRSTAT_TX_RDY)) {}
			IO_UTXH0 = '\r';
		}
	}
	if (!intmask)
		IO_INTSUBMSK &= ~INT_TXD0;
}

static int rs_console_wait_key(struct console *co)
{
	int c;
	unsigned long intmask;

	intmask = IO_INTSUBMSK & INT_RXD0;
	if (!intmask)
		IO_INTSUBMSK |= INT_RXD0;
	while (!(IO_UTRSTAT0 & UTRSTAT_RX_RDY)) {}
	c = IO_URXH0;
	if (!intmask)
		IO_INTSUBMSK &= ~INT_RXD0;
	return (c & 0xff);
}

static kdev_t rs_console_device(struct console *c)
{
	return MKDEV(SERIAL_S3C2440_MAJOR, SERIAL_S3C2440_MINOR);
}

static int __init rs_console_setup(struct console *co, char *options)
{
	IO_UCON0 = 0;
	IO_INTSUBMSK |= (INT_RXD0 | INT_TXD0);
	IO_UFCON0 = 0;
	IO_UMCON0 = 0;

	IO_UCON0 = 0x0205;
	return 0;
}

static struct console rs_cons = {
	name:		SERIAL_S3C2440_NAME,
	write:		rs_console_write,
	device:		rs_console_device,
	wait_key:	rs_console_wait_key,
	setup:		rs_console_setup,
	flags:		CON_PRINTBUFFER,
	index:		-1,
};

void __init s3c2440_rs_console_init(void)
{
	register_console(&rs_cons);
}

#endif /* CONFIG_SERIAL_S3C2440_CONSOLE */
